/*
 * A super-simple database system
 */

#include <sys/stat.h>
#include <sys/types.h>
#include <stdlib.h>
#ifndef _WIN32
#include <unistd.h>
#else
#include <io.h>
#define R_OK 4
#define W_OK 2
#define X_OK 0
#endif
#include <fcntl.h>
#include <string.h>
#include <stdio.h>

#include "db.h"
#include "db_internal.h"

/*
 * datum type strings - keep this in sync with db_datum_type_t
 */
static char *datum_type_name[] = {
  "integer",	/* DBD_INT */
  "string",	/* DBD_STR */
  "float",	/* DBD_DBL */
  "MAC"		/* DBD_MAC */
};

/*
 * static prototypes
 */
static char *duplicate_string(char *src);
static void db_free_datum_list(db_datum_t **dp, int r, int c);
static void db_free_datum_array(db_datum_t *dp, int n);
static db_datum_t *db_duplicate_datum_array(db_datum_t *dp, int n);
static void db_null_row_by_index(db_table_ptr_t tp, int  row);
static void db_remove_null_rows(db_table_ptr_t tp);


/*
 * A database is simply a directory with a file in it per table.
 * Each table is a comma-seperated-variable file, named "<table>.csv"
 * and is perfectly compatible with MS Excel or other spreadsheet.
 *
 * The first line of the file has the name of each column, the second line
 * has the type of each column, and the rest of the lines are one database row
 * each.  Example:
 *  name, rank, serial_number
 *  DBD_STR, DBD_STR, DBD_INT
 *  "John Wayne", "Actor", "12344"
 *  "One, Two", Private, -7890
 */

/*
 * utility to manage string arrays
 */
void
free_string_array(
  char **strings,
  int nstr)
{
  int i;

  if (strings != NULL) {
    for (i=0; i<nstr; ++i) {
      if (strings[i] != NULL) free(strings[i]);
    }
    free(strings);
  }
}

char **
duplicate_string_array(
  char **strings,
  int nstr)
{
  char **sp;
  int i;

  sp = (char **) calloc(nstr, sizeof(char *));
  if (sp == NULL) return NULL;

  for (i=0; i<nstr; ++i) {
    sp[i] = duplicate_string(strings[i]);
    if (sp[i] == NULL) {
      free_string_array(sp, nstr);
      return NULL;
    }
  }

  return sp;
}

/*
 * parse a line of strings of the form "aaa","nnn"
 * "s are optional if no ','s in the string
 * No "s are allowed as part of the data.
 */
#define isterm(C) (((C)=='\0')||((C)=='\n')||((C)=='\r'))
static char **
line_to_strings(
  char *line,
  int *nstr)
{
  char *lp;
  char **wp;
  int quoted;
  int w;

  wp = (char **) calloc(DB_MAX_COLS, sizeof(char *));
  if (wp == NULL) return NULL;

  w = 0;

  lp = line;
  while (!isterm(*lp)) {
    if (*lp == '"') {
      quoted = 1;
      ++lp;
      if (isterm(*lp)) goto badparse;
    } else {
      quoted = 0;
    }

    wp[w] = lp;		/* save the start of this word */
    ++w;

    /* now find the end of this word */
    if (quoted) {
      while (*lp != '"') {
        if (isterm(*lp)) goto badparse;
	++lp;
      }
      *lp = '\0';

      /* skip to either comma or end of line */
      ++lp;
      if (*lp == ',') ++lp;	/* if comma, one more to next word */

    } else {	/* not quoted */

      /* find the end of this word */
      while (*lp != ',' && !isterm(*lp)) ++lp;

      /* have to deal with trailing comma specially */
      if (*lp == ',' && isterm(lp[1])) {
        *lp++ = '\0';		/* terminate this string */
	wp[w++] = lp;		/* add a null string */
        *lp = '\0';		/* terminate it */

      } else if (*lp != '\0') {
        *lp = '\0';		/* terminate the string */
	++lp;			/* and skip to next */
      }
    } 
  }

  *nstr = w;
  return wp;
    
 badparse:
  free(wp);
  return NULL;
}


/*
 * string routines
 */
static char *
duplicate_string(
  char *src)
{
  char *new;

  /* convert NULL input into an empty string */
  if (src == NULL) src = "";

  new = malloc(strlen(src)+1);
  if (new == NULL) return NULL;

  strcpy(new, src);
  return new;
}

/*
 * Datum routines
 */
void
db_free_datum(
  db_datum_t *dp)
{
  db_free_datum_array(dp, 1);
}

static void
db_free_datum_array(
  db_datum_t *dp,
  int n)
{
  int i;

  if (dp != NULL) {
    for (i=0; i<n; ++i) {
      if (dp[i].dd_type == DBD_STR && dp[i].dbd_str != NULL) {
	free(dp[i].dbd_str);
      }
    }
    free(dp);
  }
}

/* free a list of datum arrays */
static void
db_free_datum_list(
  db_datum_t **dl,
  int r,
  int c)
{
  int i;

  if (dl != NULL) {
    for (i=0; i<r; ++i) {
      db_free_datum_array(dl[i], c);
    }
    free(dl);
  }
}

/* duplicate a an array of datum arrays */
static db_datum_t **
db_duplicate_datum_list(
  db_datum_t **dp,
  int rows,
  int cols)
{
  int r;
  db_datum_t **result;

  result = (db_datum_t **) calloc(rows, sizeof(db_datum_t *));
  if (result == NULL) goto except;

  for (r=0; r<rows; ++r) {
    result[r] = db_duplicate_datum_array(dp[r], cols);
    if (result[r] == NULL) goto except;
  }

  return result;

 except:
  if (result != NULL) db_free_datum_list(dp, rows, cols);
  return NULL;
}

/* duplicate a data array */
static db_datum_t *
db_duplicate_datum_array(
  db_datum_t *dp,
  int n)
{
  db_datum_t *new;
  int i;

  /* allocate the new array */
  new = (db_datum_t *) calloc(n, sizeof(db_datum_t));
  if (new == NULL) return NULL;

  /* and duplicate each datum */
  for (i=0; i<n; ++i) {

    new[i] = dp[i];	/* copy the datum contents */

    /* strings need special treatment */
    if (dp[i].dd_type == DBD_STR) {
      new[i].dbd_str = duplicate_string(dp[i].dbd_str);
      if (new[i].dbd_str == NULL) {
        db_free_datum_array(new, n);
	return NULL;
      }
    }
  }

  return new;
}



/*
 * destroy/allocate a database descriptor
 */
static char *
database_dirname(
  char *dir,
  char *db_name)
{
  char *dirname;

  dirname = malloc(strlen(dir) + strlen(db_name) + 2);
  if (dirname == NULL) return NULL;

  /* Fill in the fully qualified database name */
  sprintf(dirname, "%s/%s", dir, db_name);
  return dirname;
}

static void
destroy_database(
  db_database_ptr_t dbp)
{
  if (dbp != NULL) {
    if (dbp->db_dir != NULL) free(dbp->db_dir);
    free(dbp);
  }
}

static db_database_ptr_t
alloc_database(
  char *dir,
  char *db_name)
{
  db_database_ptr_t dbp;

  dbp = (db_database_ptr_t) calloc(1, sizeof(struct db_database));
  if (dbp == NULL) goto db_alloc_fail;

  /* build the full db name */
  dbp->db_dir = database_dirname(dir, db_name);
  if (dbp->db_dir == NULL) goto db_alloc_fail;

  return dbp;

 db_alloc_fail:
  if (dbp != NULL) destroy_database(dbp);
  return NULL;
}

/*
 * destroy/allocate a table descriptor
 */
static char *
table_filename(
  db_database_ptr_t dbp,
  char *tbl_name,
  char *ext)
{
  char *filename;

  filename = malloc(strlen(dbp->db_dir) + strlen(tbl_name) + strlen(ext) + 3);
  if (filename == NULL) return NULL;

  /* Fill in the fully qualified table name */
  sprintf(filename, "%s/%s.%s", dbp->db_dir, tbl_name, ext);
  return filename;
}

static void
destroy_table(
  db_table_ptr_t tp)
{
  if (tp != NULL) {
    if (tp->full_name != NULL) free(tp->full_name);
    if (tp->new_name != NULL) free(tp->new_name);
    if (tp->old_name != NULL) free(tp->old_name);
    if (tp->col_names != NULL) free_string_array(tp->col_names, tp->ncols);
    if (tp->col_types != NULL) free(tp->col_types);
    if (tp->rows) db_free_datum_list(tp->rows, tp->nrows, tp->ncols);
    free(tp);
  }
}

static db_table_ptr_t
alloc_table(
  db_database_ptr_t dbp,
  char *tbl_name)
{
  db_table_ptr_t tp;

  tp = (db_table_ptr_t) calloc(1, sizeof(struct db_table));
  if (tp == NULL) goto tbl_alloc_fail;

  tp->db = dbp;		/* reference parent DB */

  tp->full_name = table_filename(dbp, tbl_name, DB_TBLEXT_PRIMARY);
  if (tp->full_name == NULL) goto tbl_alloc_fail;

  tp->new_name = table_filename(dbp, tbl_name, DB_TBLEXT_NEW);
  if (tp->new_name == NULL) goto tbl_alloc_fail;

  tp->old_name = table_filename(dbp, tbl_name, DB_TBLEXT_OLD);
  if (tp->old_name == NULL) goto tbl_alloc_fail;

  return tp;

 tbl_alloc_fail:
  if (dbp != NULL) destroy_table(tp);
  return NULL;
}

static int
expand_table(
  db_table_ptr_t tp)
{
  db_datum_t **newrows;

  /* make room for DB_ROW_ALLOC_INCR more rows */
  tp->alloc_rows += DB_ROW_ALLOC_INCR;

  /* try to expand the table.  if it fails, make sure to leave table intact */
  newrows = (db_datum_t **) realloc(
  		tp->rows, tp->alloc_rows * sizeof(db_datum_t *));
  if (newrows == NULL) {
    return -1;
  }

  /* expansion successful */
  tp->rows = newrows;
  return 0;
}

static int
write_table_header(
  FILE *fp,
  int ncols,
  char **col_names,
  db_datum_type_t *col_types)
{
  int i;

  /* write the column headers */
  for (i=0; i<ncols; ++i) {
    if (strchr(col_names[i], ',') != NULL) {
      fprintf(fp, "\"%s\"", col_names[i]);
    } else {
      fprintf(fp, "%s", col_names[i]);
    }
    if (i<(ncols-1)) {
      putc(',', fp);
    } else {
      putc('\n', fp);
    }
  }

  /* write the data type for each column */
  for (i=0; i<ncols; ++i) {
    if (strchr(datum_type_name[col_types[i]], ',') != NULL) {
      fprintf(fp, "\"%s\"", datum_type_name[col_types[i]]);
    } else {
      fprintf(fp, "%s", datum_type_name[col_types[i]]);
    }
    if (i<(ncols-1)) {
      putc(',', fp);
    } else {
      putc('\n', fp);
    }
  }

  return 0;
}

/* write an array of n data elements */
static int
write_table_row(
  FILE *fp,
  db_datum_t *dp,
  int n)
{
  int i;

  for (i=0; i<n; ++i) {
    switch (dp[i].dd_type) {
    case DBD_STR:
      /* If there is a comma in the string, add quotes */
      if (strchr(dp[i].dbd_str, ',') != NULL) {
	fprintf(fp, "\"%s\"", dp[i].dbd_str);
      } else {
	fprintf(fp, "%s", dp[i].dbd_str);
      }
      break;
    case DBD_INT:
      fprintf(fp, "%d", dp[i].dbd_int);
      break;
    case DBD_DBL:
      fprintf(fp, "%f", dp[i].dbd_dbl);
      break;
    case DBD_MAC:
      fprintf(fp, "%02x:%02x:%02x:%02x:%02x:%02x",
        dp[i].dbd_mac[0],
        dp[i].dbd_mac[1],
        dp[i].dbd_mac[2],
        dp[i].dbd_mac[3],
        dp[i].dbd_mac[4],
        dp[i].dbd_mac[5]);
      break;
    default:
      return -1;
      break;
    }

    /* add a comma between fields */
    if (i < n-1) fprintf(fp, ",");
  }

  /* add a newline at the end */
  fprintf(fp, "\n");
  return 0;
}

/*
 * string to datum - convert ASCII into a datum
 */
static int
string_to_datum(
  db_datum_t *dp,
  char *str)
{
  switch (dp->dd_type) {
  case DBD_INT:
    dp->dbd_int = atoi(str);
    return 0;
  case DBD_STR:
    dp->dbd_str = duplicate_string(str);
    if (dp->dbd_str == NULL) {
      return -1;
    } else {
      return 0;
    }
  case DBD_DBL:
    dp->dbd_dbl = atof(str);
    return 0;
  case DBD_MAC:
    {
      int mac[6];
      int i;
      i = sscanf(str, "%2x:%2x:%2x:%2x:%2x:%2x",
      		 &mac[0], &mac[1], &mac[2], &mac[3], &mac[4], &mac[5]);
      if (i != 6) return -1;
      while (--i >= 0) {
        dp->dbd_mac[i] = mac[i];
      }
      return 0;
    }
  default:
    return -1;
  }
}

/*
 * db_create_database -
 * This just makes sure the directory a database is to reside in exists.
 * If the directory already exists or cannot be created, an error is returned.
 */
int
db_create_database(
  char *dir,
  char *db_name)
{
  char *dirname;
  int rc;

  /* build the database directory name */
  dirname = database_dirname(dir, db_name);
  if (dirname == NULL) goto db_create_fail;

  rc = mkdir(dirname, 0755);
  if (rc != 0) {
    perror(dirname);
    goto db_create_fail;
  }

  /* DB directory successfully created */
  free(dirname);
  return 0;

 db_create_fail:
  if (dirname != NULL) free(dirname);
  return -1;
}


/*
 * db_open_database -
 * This opens an existing database.  This just checks to see
 * that the container directory exists.
 */
db_database_ptr_t
db_open_database(
  char *dir,
  char *db_name,
  int write)
{
  db_database_ptr_t dbp;
  int rc;
  int mode;

  /* allocate the database descriptor */
  dbp = alloc_database(dir, db_name);
  if (dbp == NULL) goto db_open_fail;

  dbp->writeable = write;	/* save writeable status */

  /* build perms to check for, depending on whether we need to wtite */
  mode = R_OK|X_OK;
  if (write) mode |= W_OK;

  /* see if we can access the DB dir */
  rc = access(dbp->db_dir, mode);
  if (rc != 0 && errno != ENOENT) goto db_open_fail;

  /* If the dir does not exist and we are asking for write permission, 
   * try to create it. */
  if (rc != 0 && errno == ENOENT) {
    rc = mkdir(dbp->db_dir, 0755);
    if (rc != 0) goto db_open_fail;
  }

  /* DB directory exists and is accessible */
  return dbp;

 db_open_fail:
  if (dbp != NULL) destroy_database(dbp);
  return NULL;
}

/*
 * close a database - this ensures that no open tables remain on this database
 *                    and then frees any resources associated with the database
 */
int
db_close_database(
  db_database_ptr_t dbp)
{
  /* if still tables open, complain */
  if (dbp->open_tables > 0) {
    fprintf(stderr, "Cannot close database %s, %d tables still open.\n",
      dbp->db_dir, dbp->open_tables);
    return -1;
  }

  /* free the resources */
  destroy_database(dbp);
  return 0;
}

/*
 * db_create_table - create a new database table
 */
int
db_create_table(
  db_database_ptr_t dbp,
  char *table_name,
  int ncols,
  char **col_names,
  db_datum_type_t *col_types)

{
  int rc;
  int d;
  FILE *fp;
  char *filename;

  fp = NULL;

  /* build the filename */
  filename = table_filename(dbp, table_name, DB_TBLEXT_PRIMARY);
  if (filename == NULL) goto tbl_create_fail;

  /* create the table file */
  d = open(filename, O_RDWR|O_CREAT|O_EXCL, 0644);
  if (d == -1) {
    perror(filename);
    goto tbl_create_fail;
  }

  /* open it through stdio */
  fp = fdopen(d, "w");

  rc = write_table_header(fp, ncols, col_names, col_types);
  if (rc == -1) {
    goto tbl_create_fail;
  }

  /* file contains headers, that's enough for now */
  fclose(fp);
  

  /* table created successfully */
  free(filename);
  return 0;

 tbl_create_fail:
  if (filename != NULL) free(filename);
  if (fp != NULL) fclose(fp);
  return -1;
}

/*
 * db_rename_table - rename an open table to something else.  This just
 * renames the underlying file and changes the name in the table struct
 */
int
db_rename_table(
  db_table_ptr_t tp,
  char *new_table_name)
{
  db_database_ptr_t dbp;
  char *o_old_name;
  char *o_new_name;
  char *o_full_name;
  int rc;

  /* Disallow if DB not opened for writing */
  if (!tp->db->writeable) {
    fprintf(stderr, "DB %s not opened for writing, rename not allowed.\n", 
      tp->db->db_dir);
    return -1;
  }

  o_full_name = tp->full_name;
  o_new_name = tp->new_name;
  o_old_name = tp->old_name;

  dbp = tp->db;

  tp->full_name = table_filename(dbp, new_table_name, DB_TBLEXT_PRIMARY);
  if (tp->full_name == NULL) goto tbl_rename_fail;

  tp->new_name = table_filename(dbp, new_table_name, DB_TBLEXT_NEW);
  if (tp->new_name == NULL) goto tbl_rename_fail;

  tp->old_name = table_filename(dbp, new_table_name, DB_TBLEXT_OLD);
  if (tp->old_name == NULL) goto tbl_rename_fail;

  /* rename the table */
  rc = rename(o_full_name, tp->full_name);
  if (rc == -1) {
    goto tbl_rename_fail;
  }

  /* Rename any "old" files to be friendly, but don't worry about failure */
  (void) rename(o_old_name, tp->old_name);

  free(o_old_name);
  free(o_new_name);
  free(o_full_name);

  return 0;

 tbl_rename_fail:
  tp->full_name = o_full_name;
  tp->new_name = o_new_name;
  tp->old_name = o_old_name;
  return -1;
}

/*
 * db_open_table - open an existing database table and parse the rows
 *                 into memory
 */
db_table_ptr_t
db_open_table(
  db_database_ptr_t dbp,
  char *table_name)
{
  db_table_ptr_t tp;
  FILE *fp;
  char *rp;
  int nc;
  int lineno;
  char **wp;
  char buf[DB_MAX_LINE+1];
  int c;
  int t;
  int dberrno;

  /* initialize some vars */
  tp = NULL;
  fp = NULL;
  wp = NULL;
  lineno=0;

  /* allocate a table */
  tp = alloc_table(dbp, table_name);
  if (tp == NULL) DB_ERROR(errno);

  /* open the file for reading */
  fp = fopen(tp->full_name, "r");
  if (fp == NULL) DB_ERROR(errno);
  
  /*
   * Now, we are going to read the database table into memory, 1 row at a time.
   * First, we read the names of each column (first line of file)
   * Second, we read the types of each column (second line)
   * Third, we read the row data one line at a time
   */

  /* column names */
  ++lineno;
  rp = fgets(buf, DB_MAX_LINE, fp);
  if (rp == NULL) DB_ERROR(errno);
  wp = line_to_strings(buf, &tp->ncols);
  if (wp == NULL) DB_ERROR(errno);
  tp->col_names = duplicate_string_array(wp, tp->ncols);
  free(wp);
  wp = NULL;

  /* column types */
  tp->col_types = (db_datum_type_t *) malloc(sizeof(db_datum_type_t)*tp->ncols);
  if (tp->col_types == NULL) DB_ERROR(errno);

  ++lineno;
  rp = fgets(buf, DB_MAX_LINE, fp);
  if (rp == NULL) DB_ERROR(errno);
  wp = line_to_strings(buf, &nc);
  if (wp == NULL) DB_ERROR(errno);
  if (nc != tp->ncols) goto tbl_corrupt;

  /* xlate type strings to types */
  for (c=0; c<tp->ncols; ++c) {
    for (t=0; t<DBD_NUM_TYPES; ++t) {
      if (strcmp(datum_type_name[t], wp[c]) == 0) {
        tp->col_types[c] = t;
	break;
      }
    }
    if (t >= DBD_NUM_TYPES) goto tbl_corrupt;
  }
  free(wp);
  wp = NULL;

  /*
   * Now, go through the table reading is one row at a time
   */
  while ((rp = fgets(buf, DB_MAX_LINE, fp)) != NULL) {
    db_datum_t *dtp;
    int rc;

    wp = line_to_strings(buf, &nc);
    if (wp == NULL) DB_ERROR(errno);
    if (nc != tp->ncols) goto tbl_corrupt;
    ++lineno;

    /* expand the table if necessary */
    if (tp->nrows >= tp->alloc_rows) {
      rc = expand_table(tp);
      if (rc == -1) DB_ERROR(errno);
    }

    /* allocate the row */
    tp->rows[tp->nrows] = (db_datum_t *) calloc(nc, sizeof(db_datum_t));
    if (tp->rows[tp->nrows] == NULL) DB_ERROR(errno);

    /* convert each field into it's proper type */
    dtp = tp->rows[tp->nrows];
    for (c=0; c<nc; ++c) {
      dtp->dd_type = tp->col_types[c];
      rc = string_to_datum(dtp, wp[c]);
      if (rc == -1) goto tbl_corrupt;
      ++dtp;
    }

    free(wp);
    wp = NULL;
    ++tp->nrows;	/* bump row count */
  }

  /* close up file */
  fclose(fp);
  fp = NULL;

  /* table opened successfully */
  ++dbp->open_tables;
  tp->dirty = 0;
  return tp;
 
 tbl_corrupt:
  fprintf(stderr, "DB Table %s corrupt at line %d\n", tp->full_name, lineno);
  dberrno = ENODEV;

 except:
  if (tp != NULL) destroy_table(tp);
  if (fp != NULL) fclose(fp);
  if (wp != NULL) free(wp);
  errno = dberrno;
  return NULL;
}

/*
 * db_close_table - close a table.  This frees all the memory associated with
 *                  this table.
 */
void
db_close_table(
  db_table_ptr_t tp)
{
  --tp->db->open_tables;	/* reduce open table count for parent DB */

  destroy_table(tp);
}

/*
 * Flush a table out to disk
 * This is done by writing the table to "new_name", then moving the existing
 * table file to "old_name".  "new_name" is then moved to the primary filename,
 * and...  do we delete the old file?  not yet.
 */
int
db_flush_table(
  db_table_ptr_t tp)
{
  FILE *fp;
  int old_moved;
  int new_exists;
  int rc;
  int r;

  /* If not dirty, nothing to do */
  if (!tp->dirty) return 0;

  fp = NULL;
  old_moved = 0;
  new_exists = 0;

  /* make sure the database is writeable */
  if (!tp->db->writeable) {
    fprintf(stderr, "DB %s not opened for writing, flush not allowed.\n", 
      tp->db->db_dir);
    return -1;
  }

  old_moved = 0;  /* old file not moved away yet */

  /* write the table to new_name */
  fp = fopen(tp->new_name, "w");
  if (fp == NULL) goto flush_fail;
  new_exists = 1;

  /* write out the table, header first, then data */
  rc = write_table_header(fp, tp->ncols, tp->col_names, tp->col_types);
  if (rc == -1) goto flush_fail;

  /* write out each row now */
  for (r=0; r<tp->nrows; ++r) {
    if (tp->rows[r] != NULL) {
      write_table_row(fp, tp->rows[r], tp->ncols);
    }
  }

  fflush(fp);
  if (ferror(fp)) {
    perror(tp->new_name);
    goto flush_fail;
  }
  fclose(fp);
  fp = NULL;
  
  /* new file is written, try moving the old one out of the way */
  rc = rename(tp->full_name, tp->old_name);
  if (rc == -1) {
    perror(tp->old_name);
    goto flush_fail;
  }

  /* the old file is now moved away */
  old_moved = 1;

  /* now move the new file to the primary spot */
  rc = rename(tp->new_name, tp->full_name);
  if (rc == -1) {
    perror(tp->full_name);
    goto flush_fail;
  }

  /* all done, successful return */
  tp->dirty = 0;
  return 0;

 flush_fail:
  if (fp != NULL) {
    fclose(fp);
  }
  if (new_exists) {
    (void) unlink(tp->new_name);
  }
  if (old_moved) {
    (void) rename(tp->old_name, tp->full_name); /* give it our best shot */
  }
  return -1;
}


/*
 * Add a row to a table
 */
int
db_add_row(
  db_table_ptr_t tp,
  db_datum_t *row)
{
  int rc;

  /* expand the table if necessary */
  if (tp->nrows >= tp->alloc_rows) {
    rc = expand_table(tp);
    if (rc == -1) {
      perror("expanding table");
      goto add_row_fail;
    }
  }

  /* create the new row */
  tp->rows[tp->nrows] = db_duplicate_datum_array(row, tp->ncols);
  if (tp->rows[tp->nrows] == NULL) {
    perror("duplicating row data for insertion");
    goto add_row_fail;
  }

  ++tp->nrows;		/* update number of rows */
  tp->dirty = 1;

  /* all done */
  return 0;

 add_row_fail:
  return -1;
}

/*
 * Get information about a column
 */
int
db_get_column_info(			/* get the column info for a name */
  db_table_ptr_t tp,
  char *column_name,
  int *index,
  db_datum_type_t *type)
{
  int c;

  for (c=0; c<tp->ncols; ++c) {
    if (strcmp(tp->col_names[c], column_name) == 0) {
      *index = c;
      *type = tp->col_types[c];
      return 0;
    }
  }

  return -1;
}

/*
 * db_simple_query - perform a simple query
 */
int
db_simple_query(
  db_table_ptr_t tp,
  int column,
  db_datum_t *value,
  db_datum_t ***query_result,
  int *nres)
{
  int nalloc;
  int nmatch;
  db_datum_t **result;
  db_datum_t **nr;
  int r;

  *nres = 0;	/* default is no match */

  /* "match anything" is handled as a special case */
  if (value == DB_DATUM_ANY) {
    result = db_duplicate_datum_list(tp->rows, tp->nrows, tp->ncols);
    if (result == NULL) goto except;

    *nres = tp->nrows;
    *query_result = result;
    return 0;
  }

  /* sanity check this value for proper type */
  if (value->dd_type != tp->col_types[column]) {
    fprintf(stderr, "type mismatch in query\n");
    return -1;
  }

  /* set up vars */
  nalloc = 0;
  nmatch = 0;
  result = NULL;

  /*
   * The match code is slightly ugly so that we don't have to switch
   * on datum type for EVERY comparison.
   */
  switch (value->dd_type) {
  case DBD_STR:
    for (r=0; r<tp->nrows; ++r) {
      if (tp->rows[r] == NULL) continue;
      if (strcmp(tp->rows[r][column].dbd_str, value->dbd_str) == 0) {
        if (nmatch >= nalloc) {
	  int nna;

	  nna = nalloc + DB_ROW_ALLOC_INCR;
	  nr = (db_datum_t **) realloc(result, nna * sizeof(db_datum_t *));
	  if (nr == NULL) goto except;
	  result = nr;
	  nalloc = nna;
	}

	/* save this row */
	result[nmatch] = db_duplicate_datum_array(tp->rows[r], tp->ncols);
	if (result[nmatch] == NULL) goto except;
	++nmatch;
      }
    }
    break;

  case DBD_INT:
    for (r=0; r<tp->nrows; ++r) {
      if (tp->rows[r] == NULL) continue;
      if (tp->rows[r][column].dbd_int == value->dbd_int) {
        if (nmatch >= nalloc) {
	  int nna;

	  nna = nalloc + DB_ROW_ALLOC_INCR;
	  nr = (db_datum_t **) realloc(result, nna * sizeof(db_datum_t *));
	  if (nr == NULL) goto except;
	  result = nr;
	  nalloc = nna;
	}

	/* save this row */
	result[nmatch] = db_duplicate_datum_array(tp->rows[r], tp->ncols);
	if (result[nmatch] == NULL) goto except;
	++nmatch;
      }
    }
    break;

  case DBD_MAC:
    for (r=0; r<tp->nrows; ++r) {
      if (tp->rows[r] == NULL) continue;
      if (memcmp(tp->rows[r][column].dbd_mac, value->dbd_mac, 6) == 0) {
        if (nmatch >= nalloc) {
	  int nna;

	  nna = nalloc + DB_ROW_ALLOC_INCR;
	  nr = (db_datum_t **) realloc(result, nna * sizeof(db_datum_t *));
	  if (nr == NULL) goto except;
	  result = nr;
	  nalloc = nna;
	}

	/* save this row */
	result[nmatch] = db_duplicate_datum_array(tp->rows[r], tp->ncols);
	++nmatch;
      }
    }
    break;

  default:
    fprintf(stderr, "Query on %s not supported.\n",
            datum_type_name[value->dd_type]);
    goto except;
  }

  /* trim the array to the number of results */
  if (result != NULL) {
    nr = (db_datum_t **) realloc(result, nmatch*sizeof(db_datum_t *));
    if (nr == NULL) goto except;
    result = nr;
  }

  /* return query results */
  *nres = nmatch;
  *query_result = result;
  return 0;

 except:
  if (result) db_free_datum_list(result, nalloc, tp->ncols);
  return -1;
}

/*
 * db_simple_or_query - perform a simple OR query
 */
int
db_simple_or_query(
  db_table_ptr_t tp,
  int column,
  db_datum_t *values,
  int num_vals,
  db_datum_t ***query_result,
  int *nres)
{
  int nalloc;
  int nmatch;
  db_datum_t **result;
  db_datum_t **nr;
  int v;
  int r;

  *nres = 0;	/* default is no match */

  /* sanity check this value for proper type */
  if (values[0].dd_type != tp->col_types[column]) {
    fprintf(stderr, "type mismatch in query\n");
    return -1;
  }

  /* set up vars */
  nalloc = 0;
  nmatch = 0;
  result = NULL;

  /*
   * The match code is slightly ugly so that we don't have to switch
   * on datum type for EVERY comparison.
   */
  switch (values[0].dd_type) {
  case DBD_STR:
    for (r=0; r<tp->nrows; ++r) {
      if (tp->rows[r] == NULL) continue;
      for (v=0; v<num_vals; ++v) {
	if (strcmp(tp->rows[r][column].dbd_str, values[v].dbd_str) == 0) {
	  if (nmatch >= nalloc) {
	    int nna;

	    nna = nalloc + DB_ROW_ALLOC_INCR;
	    nr = (db_datum_t **) realloc(result, nna * sizeof(db_datum_t *));
	    if (nr == NULL) goto except;
	    result = nr;
	    nalloc = nna;
	  }

	  /* save this row */
	  result[nmatch] = db_duplicate_datum_array(tp->rows[r], tp->ncols);
	  if (result[nmatch] == NULL) goto except;
	  ++nmatch;

	  break;
	}
      }
    }
    break;

  case DBD_INT:
    for (r=0; r<tp->nrows; ++r) {
      if (tp->rows[r] == NULL) continue;
      for (v=0; v<num_vals; ++v) {
	if (tp->rows[r][column].dbd_int == values[v].dbd_int) {
	  if (nmatch >= nalloc) {
	    int nna;

	    nna = nalloc + DB_ROW_ALLOC_INCR;
	    nr = (db_datum_t **) realloc(result, nna * sizeof(db_datum_t *));
	    if (nr == NULL) goto except;
	    result = nr;
	    nalloc = nna;
	  }

	  /* save this row */
	  result[nmatch] = db_duplicate_datum_array(tp->rows[r], tp->ncols);
	  if (result[nmatch] == NULL) goto except;
	  ++nmatch;

	  break;
	}
      }
    }
    break;

  case DBD_MAC:
    for (r=0; r<tp->nrows; ++r) {
      if (tp->rows[r] == NULL) continue;
      for (v=0; v<num_vals; ++v) {
	if (memcmp(tp->rows[r][column].dbd_mac, values[v].dbd_mac, 6) == 0) {
	  if (nmatch >= nalloc) {
	    int nna;

	    nna = nalloc + DB_ROW_ALLOC_INCR;
	    nr = (db_datum_t **) realloc(result, nna * sizeof(db_datum_t *));
	    if (nr == NULL) goto except;
	    result = nr;
	    nalloc = nna;
	  }

	  /* save this row */
	  result[nmatch] = db_duplicate_datum_array(tp->rows[r], tp->ncols);
	  ++nmatch;

	  break;
	}
      }
    }
    break;

  default:
    fprintf(stderr, "Query on %s not supported.\n",
            datum_type_name[values[0].dd_type]);
    goto except;
  }

  /* trim the array to the number of results */
  if (result != NULL) {
    nr = (db_datum_t **) realloc(result, nmatch*sizeof(db_datum_t *));
    if (nr == NULL) goto except;
    result = nr;
  }

  /* return query results */
  *nres = nmatch;
  *query_result = result;
  return 0;

 except:
  if (result) db_free_datum_list(result, nalloc, tp->ncols);
  return -1;
}

/*
 * db_row_query - return all rows matching the query row
 */
int
db_row_query(
  db_table_ptr_t tp,
  db_datum_t *values,
  db_datum_t ***query_result,
  int *nres)
{
  int nalloc;
  int nmatch;
  db_datum_t **result;
  db_datum_t **nr;
  int r;
  int c;

  *nres = 0;	/* default is no match */

  /* sanity check the values for proper type */
  for (c=0; c<tp->ncols; ++c) {
    if (values[c].dd_type != tp->col_types[c]
	&& values[c].dd_type != DBD_MATCHALL) {
      fprintf(stderr, "type mismatch in query in column %d\n", c);
      return -1;
    }
  }

  /* set up vars */
  nalloc = 0;
  nmatch = 0;
  result = NULL;

  for (r=0; r<tp->nrows; ++r) {
    db_datum_t *row;
    int match;

    row = tp->rows[r];		/* pointer for this row */
    match = 1;			/* default is match */

    /* check every column for match */
    for (c=0; c<tp->ncols; ++c) {

      switch (values[c].dd_type) {
      case DBD_STR:
	if (strcmp(row[c].dbd_str, values[c].dbd_str) != 0) {
	  match = 0;
	}
	break;

      case DBD_INT:
	if (row[c].dbd_int != values[c].dbd_int) {
	  match = 0;
	}
	break;

      case DBD_MAC:
	if (memcmp(row[c].dbd_mac, values[c].dbd_mac, 6) != 0) {
	  match = 0;
	}

      case DBD_MATCHALL:	/* this type matches anything */
	break;

      default:
	fprintf(stderr, "Query on %s not supported.\n",
		datum_type_name[values[c].dd_type]);
	goto except;
      }
    }

    /* If we matched all columns, save this row */
    if (match) {

      /* make more space for results if necessary */
      if (nmatch >= nalloc) {
	int nna;

	nna = nalloc + DB_ROW_ALLOC_INCR;
	nr = (db_datum_t **) realloc(result, nna * sizeof(db_datum_t *));
	if (nr == NULL) goto except;
	result = nr;
	nalloc = nna;
      }

      /* save this row */
      result[nmatch] = db_duplicate_datum_array(tp->rows[r], tp->ncols);
      if (result[nmatch] == NULL) goto except;
      ++nmatch;
    }
  }

  /* trim the array to the number of results */
  if (result != NULL) {
    nr = (db_datum_t **) realloc(result, nmatch*sizeof(db_datum_t *));
    if (nr == NULL) goto except;
    result = nr;
  }

  /* return query results */
  *nres = nmatch;
  *query_result = result;
  return 0;

 except:
  if (result) db_free_datum_list(result, nalloc, tp->ncols);
  return -1;
}

/*
 * db_row_query - return all rows matching one of the query rows
 */
int
db_row_or_query(
  db_table_ptr_t tp,
  db_datum_t **values,
  int num_rows,
  db_datum_t ***query_result,
  int *nres)
{
  int nalloc;
  int nmatch;
  db_datum_t **result;
  db_datum_t **nr;
  int v;
  int r;
  int c;

  *nres = 0;	/* default is no match */

  /* sanity check the values for proper type */
  for (c=0; c<tp->ncols; ++c) {
    if (values[0][c].dd_type != tp->col_types[c]
	&& values[0][c].dd_type != DBD_MATCHALL) {
      fprintf(stderr, "type mismatch in query in column %d\n", c);
      return -1;
    }
  }

  /* set up vars */
  nalloc = 0;
  nmatch = 0;
  result = NULL;

  for (r=0; r<tp->nrows; ++r) {
    db_datum_t *row;
    int match;

    row = tp->rows[r];		/* pointer for this row */
    match = 1;			/* default is match */

    /* check every column for match */
    for (c=0; c<tp->ncols; ++c) {
      int colmatch;

      colmatch = 0;		/* default is column does not match */

      switch (values[0][c].dd_type) {
      case DBD_STR:
	for (v=0; v<num_rows; ++v) {
	  if (strcmp(row[c].dbd_str, values[v][c].dbd_str) == 0) {
	    colmatch = 1;
	    break;
	  }
	}
	if (!colmatch) match = 0;
	break;

      case DBD_INT:
	for (v=0; v<num_rows; ++v) {
	  if (row[c].dbd_int == values[v][c].dbd_int) {
	    colmatch = 1;
	    break;
	  }
	}
	if (!colmatch) match = 0;
	break;

      case DBD_MAC:
	for (v=0; v<num_rows; ++v) {
	  if (memcmp(row[c].dbd_mac, values[v][c].dbd_mac, 6) == 0) {
	    colmatch = 1;
	    break;
	  }
	}
	if (!colmatch) match = 0;
	break;

      case DBD_MATCHALL:	/* this type matches anything */
	break;

      default:
	fprintf(stderr, "Query on %s not supported.\n",
		datum_type_name[values[0][c].dd_type]);
	goto except;
      }
    }

    /* If we matched all columns, save this row */
    if (match) {

      /* make more space for results if necessary */
      if (nmatch >= nalloc) {
	int nna;

	nna = nalloc + DB_ROW_ALLOC_INCR;
	nr = (db_datum_t **) realloc(result, nna * sizeof(db_datum_t *));
	if (nr == NULL) goto except;
	result = nr;
	nalloc = nna;
      }

      /* save this row */
      result[nmatch] = db_duplicate_datum_array(tp->rows[r], tp->ncols);
      if (result[nmatch] == NULL) goto except;
      ++nmatch;
    }
  }

  /* trim the array to the number of results */
  if (result != NULL) {
    nr = (db_datum_t **) realloc(result, nmatch*sizeof(db_datum_t *));
    if (nr == NULL) goto except;
    result = nr;
  }

  /* return query results */
  *nres = nmatch;
  *query_result = result;
  return 0;

 except:
  if (result) db_free_datum_list(result, nalloc, tp->ncols);
  return -1;
}

/*
 * db_delete_row - delete all rows matching this input pattern
 */
int
db_delete_row(
  db_table_ptr_t tp,
  db_datum_t *values)
{
  int r;
  int c;

  /* sanity check the values for proper type */
  for (c=0; c<tp->ncols; ++c) {
    if (values[c].dd_type != tp->col_types[c]
	&& values[c].dd_type != DBD_MATCHALL) {
      fprintf(stderr, "type mismatch in delete pattern in column %d\n", c);
      return -1;
    }
  }

  for (r=0; r<tp->nrows; ++r) {
    db_datum_t *row;
    int match;

    row = tp->rows[r];		/* pointer for this row */
    if (row == NULL) continue;

    match = 1;			/* default is match */

    /* check every column for match */
    for (c=0; c<tp->ncols; ++c) {

      switch (values[c].dd_type) {
      case DBD_STR:
	if (strcmp(row[c].dbd_str, values[c].dbd_str) != 0) {
	  match = 0;
	}
	break;

      case DBD_INT:
	if (row[c].dbd_int != values[c].dbd_int) {
	  match = 0;
	}
	break;

      case DBD_MAC:
	if (memcmp(row[c].dbd_mac, values[c].dbd_mac, 6) != 0) {
	  match = 0;
	}

      case DBD_MATCHALL:	/* this type matches anything */
	break;

      default:
	fprintf(stderr, "Matching on %s not supported.\n",
		datum_type_name[values[c].dd_type]);
	goto except;
      }
    }

    /* If we matched all columns, delete this row */
    if (match) {
      db_null_row_by_index(tp, r);
    }
  }

  db_remove_null_rows(tp);	/* clean up all the NULLed rows */
  tp->dirty = 1;

  return 0;

 except:
  db_remove_null_rows(tp);	/* clean up any NULLed rows */
  return -1;
}

/*
 * Delete a row from a table
 * This leaves a NULL pointer in the table slot, db_remove_null_rows
 * should be called after all desired deleting is done.
 *
 * XXX - rest of codeo ought to deal with NULL rows
 */
static void
db_null_row_by_index(
  db_table_ptr_t tp,
  int r)
{
  /* sanity check row index */
  if (r >= tp->nrows) return;

  /* free the memory from this row */
  db_free_datum_array(tp->rows[r], tp->ncols);

  /* mark this row as gone */
  tp->rows[r] = NULL;
}

/*
 * Remove all NULL rows from a table
 */
static void
db_remove_null_rows(
  db_table_ptr_t tp)
{
  int r;
  int nr;

  nr = 0;
  for (r=0; r<tp->nrows; ++r) {
    if (tp->rows[r] != NULL) {
      if (nr != r) {
	tp->rows[nr] = tp->rows[r];
      }
      ++nr;
    }
  }

  tp->nrows = nr;
}

/*
 * free a query result
 */
void
db_free_query_res(
  db_table_ptr_t tp,
  db_datum_t **res,
  int nres)
{
  if (res == NULL) return;

  db_free_datum_list(res, nres, tp->ncols);
}

/*
 * Return the number of columns in a table
 */
int
db_tbl_num_cols(
  db_table_ptr_t tp)
{
  return tp->ncols;
}

/*
 * Return an array of data with count and types matching a row
 * in the specified table suitable for use as a template for adding
 * more rows.
 */
db_datum_t *
db_alloc_row_template(
  db_table_ptr_t table,
  int *ncols)
{
  db_datum_t *dbp;
  int i;

  /* allocate the new array */
  dbp = (db_datum_t *) calloc(table->ncols, sizeof(db_datum_t));
  if (dbp == NULL) return NULL;

  /* fill in type for each datum */
  for (i=0; i<table->ncols; ++i) {
    dbp[i].dd_type = table->col_types[i];
  }

  /* if requested, return number of columns */
  if (ncols != NULL) *ncols = table->ncols;

  return dbp;
}
